/* eslint no-console:0 */
'use strict';

/**
 * @typedef {Object} Edition
 * @property {string} description
 * @property {string} directory
 * @property {string} entry
 * @property {Array<string>} syntaxes
 * @property {false|Object.<string, string|boolean>} engines
 * @public
 */

/**
 * @typedef {Object} Options
 * @property {Function} require - the require method of the calling module, used to ensure require paths remain correct
 * @property {string} [packagePath] - if provided, this is used for debugging
 * @property {boolean} [verbose] - if provided, any error loading an edition will be logged. By default, errors are only logged if all editions failed. If not provided, process.env.EDITIONS_VERBOSE is used.
 * @property {string} [cwd] - if provided, this will be the cwd for entries
 * @property {string} [entry] - if provided, should be a relative path to the entry point of the edition
 * @property {string} [package] - if provided, should be the name of the package that we are loading the editions for
 * @property {stream.Writable} [stderr] - if not provided, will use process.stderr instead. It is the stream that verbose errors are logged to.
 * @public
 */

// Imports

var pathUtil = require('path');
var semver = require('semver');
var Errlop = require('errlop');

// Helpers
function stringify(value) {
	return typeof value === 'string' ? value : JSON.stringify(value);
}

// Environment fetching
var BLACKLIST = process.env.EDITIONS_SYNTAX_BLACKLIST && process.env.EDITIONS_SYNTAX_BLACKLIST.split(/[, ]+/g);
var NODE_VERSION = process.versions.node;
var VERBOSE = process.env.EDITIONS_VERBOSE === true || process.env.EDITIONS_VERBOSE === 'yes' || process.env.EDITIONS_VERBOSE === 'true' || false;

// Cache of which syntax combinations are supported or unsupported
// Object.<string, Error>
var blacklist = {};

// Check the environment configuration for a syntax blacklist
if (BLACKLIST) {
	for (var i = 0; i < BLACKLIST.length; ++i) {
		var syntax = BLACKLIST[i].trim().toLowerCase();
		blacklist[syntax] = new Errlop('The EDITIONS_SYNTAX_BLACKLIST environment variable blacklisted the syntax [' + syntax + ']');
	}
}

// Blacklist the syntax 'esnext' if our node version is below 0.12
if (semver.satisfies(NODE_VERSION, '<0.12')) {
	blacklist.esnext = new Error('The esnext syntax is skipped on early node versions as attempting to use esnext features will output debugging information on these node versions');
}

/**
 * Attempt to load a specific edition
 * @param {Edition} edition
 * @param {Options} opts
 * @returns {*} the result of the loaded edition
 * @throws {Error} an error if the edition failed to load
 * @public
 */
function requireEdition(edition, opts) {
	// Verify the edition is valid
	if (!edition.description || !edition.directory || !edition.entry || edition.engines == null) {
		var editionInvalidError = new Errlop('Each edition must have its [description, directory, entry, engines] fields defined, yet all it had was [' + Object.keys(edition).join(', ') + ']');
		editionInvalidError.level = 'fatal';
		throw editionInvalidError;
	}

	// Verify engine support
	if (edition.engines === false) {
		throw new Errlop('Skipping edition [' + edition.description + '] because its engines field was false');
	}
	if (!edition.engines.node) {
		throw new Errlop('Skipping edition [' + edition.description + '] because its .engines.node field was falsey');
	}
	if (edition.engines.node !== true && semver.satisfies(NODE_VERSION, edition.engines.node) === false) {
		throw new Errlop('Skipping edition [' + edition.description + '] because our current node version [' + NODE_VERSION + '] is not supported by its [' + stringify(edition.engines.node) + ']');
	}

	// Verify syntax support
	// Convert syntaxes into a sorted lowercase string
	var syntaxes = edition.syntaxes && edition.syntaxes.map(function (i) {
		return i.toLowerCase();
	}).sort() || [];
	for (var index = 0; index < syntaxes.length; index++) {
		var _syntax = syntaxes[index];
		var blacklisted = blacklist[_syntax];
		if (blacklisted) {
			throw new Errlop('Skipping edition [' + edition.description + '] because it contained a blacklisted syntax [' + _syntax + ']', blacklisted);
		}
	}

	// Load the edition
	var entry = pathUtil.resolve(opts.cwd || '', edition.directory, opts.entry || edition.entry);
	try {
		return opts.require(entry);
	} catch (loadError) {
		// Note the error with more details
		throw new Errlop('Skipped edition [' + edition.description + '] at entry [' + entry + '] because it failed to load', loadError);
	}
}

/**
 * Cycles through a list of editions, returning the first suitable edition that it was able to load
 * @param {Array<Edition>} editions
 * @param {Options} opts
 * @returns {*} the result of the loaded edition
 * @throws {Error} an error if a suitable edition was unable to be resolved
 * @public
 */
function requireEditions(editions, opts) {
	// Check
	if (!editions || editions.length === 0) {
		if (opts.packagePath) {
			throw new Errlop('There were no editions specified for package [' + opts.packagePath + ']');
		} else {
			throw new Errlop('There were no editions specified');
		}
	}

	// Note the last error message
	var result = void 0,
	    editionsError = null,
	    loaded = false;

	// Cycle through the editions
	for (var _i = 0; _i < editions.length; ++_i) {
		var edition = editions[_i];
		try {
			result = requireEdition(edition, opts);
			loaded = true;
			break;
		} catch (editionError) {
			if (editionError.level === 'fatal') {
				editionsError = editionError;
				break;
			} else if (editionsError) {
				editionsError = new Errlop(editionsError, editionError);
			} else {
				editionsError = editionError;
			}
		}
	}

	if (loaded) {
		var verbose = opts.verbose == null ? VERBOSE : opts.verbose;
		if (editionsError && verbose) {
			var stderr = opts.stderr || process.stderr;
			stderr.write(editionsError.stack + '\n');
		}
		return result;
	} else if (editionsError) {
		if (opts.packagePath) {
			throw new Errlop('There were no suitable editions for package [' + opts.packagePath + ']', editionsError);
		} else {
			throw new Errlop('There were no suitable editions', editionsError);
		}
	}
}

/**
 * Cycle through the editions for a package and require the correct one
 * @param {Options.cwd} cwd
 * @param {Options.require} require
 * @param {Options.entry} [entry]
 * @returns {*} the result of the loaded edition
 * @throws {Error} an error if a suitable edition was unable to be resolved
 * @public
 */
function requirePackage(cwd, require, entry) {
	// Load the package.json file to fetch `name` for debugging and `editions` for loading
	var packagePath = pathUtil.resolve(cwd, 'package.json');

	var _require = require(packagePath),
	    editions = _require.editions;

	var opts = { packagePath: packagePath, cwd: cwd, require: require, entry: entry };
	return requireEditions(editions, opts);
}

// Exports
module.exports = { requireEdition: requireEdition, requireEditions: requireEditions, requirePackage: requirePackage };